#
#Licensed to the Apache Software Foundation (ASF) under one
#or more contributor license agreements.  See the NOTICE file
#distributed with this work for additional information
#regarding copyright ownership.  The ASF licenses this file
#to you under the Apache License, Version 2.0 (the
#"License"); you may not use this file except in compliance
#with the License.  You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
#Unless required by applicable law or agreed to in writing,
#software distributed under the License is distributed on an
#"AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
#KIND, either express or implied.  See the License for the
#specific language governing permissions and limitations
#under the License.
#
require "logstash/outputs/base"
require "logstash/namespace"
require 'stud/buffer'
require "maxctunnel_jars"

# Datahub output that does nothing.
class LogStash::Outputs::MaxCTunnel < LogStash::Outputs::Base
    include Stud::Buffer
    config_name "maxctunnel"

    config :aliyun_access_id, :validate => :string, :required => true
    config :aliyun_access_key, :validate => :string, :required => true
    config :aliyun_mc_endpoint, :validate => :string, :required => true
    config :aliyun_mc_tunnel_endpoint, :validate => :string, :required => false, :default => ""
    config :project, :validate => :string, :required => true
    config :table, :validate => :string, :required => true
    config :retry_time, :validate => :number, :required => false, :default => 3
    config :retry_interval, :validate => :number, :required => false, :default => 1
    config :batch_size, :validate => :number, :required => false, :default => 100
    config :batch_timeout, :validate => :number, :required => false, :default => 5
    config :value_fields, :validate => :array, :required => false, :default => []
    config :partition, :validate => :string, :required => false, :default => nil
    config :partition_time_format, :validate => :string, :required => false, :default => nil

    OdpsTunnelPackage = com.aliyun.odps.tunnel
    OdpsPackage = com.aliyun.odps
    OdpsAccountPackage = com.aliyun.odps.account
    OdpsDataPackage = com.aliyun.odps.data

    public
    def register
        begin
            @account = OdpsAccountPackage::AliyunAccount::new(@aliyun_access_id, @aliyun_access_key)
            @odps = OdpsPackage.Odps::new(@account)
            @odps.setEndpoint(@aliyun_mc_endpoint)
            @odps.setDefaultProject(@project)
            @tunnel = OdpsTunnelPackage::TableTunnel::new(@odps)
            unless @aliyun_mc_tunnel_endpoint.empty?
                @tunnel.setEndpoint(@aliyun_mc_tunnel_endpoint)
            end

            buffer_initialize(
                    :max_items => @batch_size,
                    :max_interval => @batch_timeout,
                    :logger => @logger
            )
        rescue => e
            @logger.error "Init failed!"  + e.message + " " + e.backtrace.inspect.to_s
            raise e
        end
    end # def register

    public
    def receive(event)
        if event == LogStash::SHUTDOWN
            return
        end
        begin
            @logger.debug "receive data:" + event.to_json
            if @partition == nil
                buffer_receive(event, "")
            else
                #split partition
                partition_arrays = @partition.split(',')
                partitionSpecStr = ""
                eventHash = event.to_hash
                for p in partition_arrays do
                    if p.include? "strftime"
                        key=p[p.index("<")+1 .. p.index(".strftime")-1]
                        partition_column=p[0 .. p.index("=")-1]
                        timeFormat=p[p.index("(")+2 .. p.index(")")-2]
                        @logger.debug "determined: key " + key + " partcol " + partition_column + " timefmt " + timeFormat
                        if eventHash.has_key?(key)
                            if @partition_time_format == nil
                                partition_value=Time.parse(eventHash[key].to_s).strftime(timeFormat)
                            else
                                partition_value=Time.strptime(eventHash[key].to_s, @partition_time_format).strftime(timeFormat)
                            end
                            partitionSpecStr += partition_column + "='" + partition_value + "',"
                        else
                            raise "partition has no corresponding source key or the partition expression is wrong for time reformatting,"+p.to_s
                        end
                    elsif p.include? "=$<"
                        key=p[p.index("<")+1 .. p.index(">")-1]
                        partition_column=p[0 .. p.index("=")-1]
                        if eventHash.has_key?(key)
                            partition_value=eventHash[key]
                            partitionSpecStr += partition_column + "='" + partition_value + "',"
                        else
                            raise "partition has no corresponding source key or the partition expression is wrong for identical,"+p.to_s
                        end
                    else
                        partitionSpecStr += p + ","
                    end
                end
                partitionSpecStr = partitionSpecStr.chop
                @logger.debug "partition:" + partitionSpecStr
                buffer_receive(event, partitionSpecStr)
            end
        rescue => e
            @logger.error "receive event failed:" + e.message + " " + e.backtrace.inspect.to_s
        end
    end # def event

    # called from Stud::Buffer#buffer_flush when there are events to flush
    def flush(eventList, partitionSpecStr, close=false)
        retry_time = @retry_time
        partitionSpec = nil
        if @partition != nil
            partitionSpec = OdpsPackage::PartitionSpec::new(partitionSpecStr)
            @logger.debug "generated partitionSpec" + partitionSpec.toString()
        end
        @logger.debug "trying to write " + eventList.size.to_s() + " records on partition " + partitionSpecStr + " ..."
        begin
            uploadSession = nil
            if partitionSpec != nil
                uploadSession = @tunnel.createStreamUploadSession(@project, @table, partitionSpec, true) # auto creat partition
            else
                uploadSession = @tunnel.createStreamUploadSession(@project, @table)
            end
            schema = uploadSession.getSchema()
            @logger.debug "Schema:" + schema.to_s
            pack = uploadSession.newRecordPack(OdpsTunnelPackage.io.CompressOption::new())
            eventList.each do |event|
                @logger.debug "event" + event.to_hash.to_s
                eventData = event.to_hash
                if value_fields.size != schema.getColumns().size()
                    raise "value fields count doesnt match with schema cols."
                end
                record = uploadSession.newRecord()
                for index in 0...schema.getColumns().size() do
                    begin
                        col = schema.getColumn(index)
                        @logger.debug "data:" + eventData[value_fields[index]].to_s.chomp + " colName:" + col.getName
                        if (col.getType() == OdpsPackage.OdpsType::STRING)
                            record.setString(index, eventData[value_fields[index]].to_s.chomp)
                        elsif (col.getType() == OdpsPackage.OdpsType::BIGINT)
                            record.setBigint(index, eventData[value_fields[index]].to_i)
                        elsif (col.getType() == OdpsPackage.OdpsType::DOUBLE)
                            record.setDouble(index, eventData[value_fields[index]].to_f)
                        elsif (col.getType() == OdpsPackage.OdpsType::DATETIME)
                            datetime = Time.parse(eventData[value_fields[index]].to_s).to_i*1000
                            record.setDatetime(index, java.util.Date::new(datetime))
                        elsif (col.getType() == OdpsPackage.OdpsType::BOOLEAN)
                            record.setBoolean(index, eventData[value_fields[index]].to_s.downcase == "true")
                        end
                    rescue => e
                        @logger.warn "Parse failure " + e.message + " happened at " + eventData.to_s
                        raise e
                    end
                end
                pack.append(record)
            end
            traceId = pack.flush()
            @logger.info "write " + eventList.size.to_s() + " records on table " + @project + "." + @table + " partition " + partitionSpecStr + " completed. TraceId: " + traceId.to_s()
        rescue OdpsPackage.OdpsException => e
            @logger.error "write data failed with odps error: " + e.message + " " + e.backtrace.inspect.to_s
            if retry_time > 0
                retry_time -= 1
                @logger.warn "write data will retry in: " + @retry_interval.to_s + " second."
                sleep(@retry_interval)
                retry
            end
            @logger.error "retry failed, drop this pack. Partition: " + partitionSpecStr + " Record count:" + eventList.size.to_s
        rescue => e
            @logger.error "write data failed with local error: " + e.message + " " + e.backtrace.inspect.to_s
            @logger.error "drop this pack. Partition:" + partitionSpecStr + " Record count:" + eventList.size.to_s
        end
    end

    def close
        buffer_flush(:final => true)
    end
end
